<?php
// Start output buffering to prevent any output before PDF generation
ob_start();

session_start();
require_once '../config/db.php';
require_once '../includes/auth_check.php';
require_once '../vendor/autoload.php';

// Check if user has permission to export reports
if (!hasPermission('export_reports') && !$isSuperAdmin) {
    ob_end_clean(); // Clean buffer before redirect
    $_SESSION['error'] = "You don't have permission to export reports.";
    header('Location: reports.php');
    exit;
}

// Get date range from request
$start_date = $_GET['start_date'] ?? date('Y-m-d', strtotime('-30 days'));
$end_date = $_GET['end_date'] ?? date('Y-m-d');

// Get store information
$stmt = $pdo->prepare("SELECT setting_key, setting_value, setting_binary FROM store_settings WHERE setting_key IN ('store_name', 'store_address', 'store_phone', 'store_email', 'store_website', 'store_logo', 'currency')");
$stmt->execute();
$store_settings = [];
$store_logo_binary = null;
$store_logo_filename = null;
while ($row = $stmt->fetch()) {
    $store_settings[$row['setting_key']] = $row['setting_value'];
    if ($row['setting_key'] === 'store_logo' && $row['setting_binary']) {
        $store_logo_binary = $row['setting_binary'];
        $store_logo_filename = $row['setting_value'];
    }
}

$storeName = $store_settings['store_name'] ?? 'Store Name';
$storeAddress = $store_settings['store_address'] ?? 'Store Address';
$storePhone = $store_settings['store_phone'] ?? 'Store Phone';
$storeEmail = $store_settings['store_email'] ?? 'Store Email';
$storeWebsite = $store_settings['store_website'] ?? '';
$storeLogo = $store_settings['store_logo'] ?? '';
$currency = $store_settings['currency'] ?? 'Ksh';

// Create temporary logo file if logo exists in database
$tempLogoPath = null;
if ($store_logo_binary && $store_logo_filename) {
    $tempDir = sys_get_temp_dir();
    $tempLogoPath = $tempDir . '/' . uniqid('logo_') . '_' . $store_logo_filename;
    
    // Check if we have required image extensions
    $hasImagick = extension_loaded('imagick');
    $hasGD = extension_loaded('gd');
    
    if ($hasImagick || $hasGD) {
        // We have image processing capabilities
        if (file_put_contents($tempLogoPath, $store_logo_binary) !== false) {
            // File created successfully
            // Check if it's a PNG with alpha channel and convert if needed
            if (strtolower(pathinfo($store_logo_filename, PATHINFO_EXTENSION)) === 'png') {
                try {
                    if ($hasImagick) {
                        // Use Imagick to convert PNG to JPEG (removes alpha channel)
                        $imagick = new Imagick($tempLogoPath);
                        $imagick->setImageBackgroundColor('white');
                        $imagick->setImageAlphaChannel(Imagick::ALPHACHANNEL_REMOVE);
                        $imagick->setImageFormat('jpeg');
                        $imagick->writeImage($tempLogoPath);
                        $imagick->clear();
                        $imagick->destroy();
                    } elseif ($hasGD) {
                        // Use GD to convert PNG to JPEG
                        $image = imagecreatefrompng($tempLogoPath);
                        if ($image !== false) {
                            // Create a white background
                            $width = imagesx($image);
                            $height = imagesy($image);
                            $newImage = imagecreatetruecolor($width, $height);
                            $white = imagecolorallocate($newImage, 255, 255, 255);
                            imagefill($newImage, 0, 0, $white);
                            
                            // Copy the PNG onto the white background
                            imagecopy($newImage, $image, 0, 0, 0, 0, $width, $height);
                            
                            // Save as JPEG
                            imagejpeg($newImage, $tempLogoPath, 90);
                            
                            // Clean up
                            imagedestroy($image);
                            imagedestroy($newImage);
                        }
                    }
                } catch (Exception $e) {
                    // If conversion fails, remove the file and continue without logo
                    if (file_exists($tempLogoPath)) {
                        unlink($tempLogoPath);
                    }
                    $tempLogoPath = null;
                    error_log("Failed to convert PNG logo: " . $e->getMessage());
                }
            }
        } else {
            $tempLogoPath = null; // Reset if failed
        }
    } else {
        // No image processing extensions available, skip logo
        $tempLogoPath = null;
        error_log("No image processing extensions (GD or Imagick) available for logo handling");
    }
}

// Get sales data
$stmt = $pdo->prepare("
    SELECT 
        s.id,
        s.created_at,
        s.total_amount,
        s.payment_method,
        s.status,
        COALESCE(c.name, 'Walk-in Customer') as customer_name,
        COALESCE(c.phone, 'N/A') as customer_phone,
        u.username as cashier
    FROM sales s 
    LEFT JOIN customers c ON s.customer_id = c.id 
    LEFT JOIN users u ON s.user_id = u.id
    WHERE DATE(s.created_at) BETWEEN ? AND ? 
    ORDER BY s.created_at DESC
");
$stmt->execute([$start_date, $end_date]);
$sales = $stmt->fetchAll();

// Get sales statistics
$stmt = $pdo->prepare("
    SELECT 
        COUNT(*) as total_transactions,
        SUM(total_amount) as total_revenue,
        AVG(total_amount) as average_sale,
        SUM(CASE WHEN payment_method = 'cash' THEN total_amount ELSE 0 END) as cash_sales,
        SUM(CASE WHEN payment_method = 'mpesa' THEN total_amount ELSE 0 END) as mpesa_sales,
        SUM(CASE WHEN payment_method = 'credit' THEN total_amount ELSE 0 END) as credit_sales
    FROM sales 
    WHERE DATE(created_at) BETWEEN ? AND ?
");
$stmt->execute([$start_date, $end_date]);
$stats = $stmt->fetch();

// Create PDF using TCPDF
class MYPDF extends TCPDF {
    public function Header() {
        global $storeName, $storeAddress, $storePhone, $storeEmail, $storeWebsite, $tempLogoPath, $start_date, $end_date;
        
        // Report title first
        $this->SetFont('helvetica', 'B', 18);
        $this->Cell(0, 12, 'SALES REPORT', 0, 1, 'C');
        $this->Ln(3);
        
        // Store logo (if exists)
        if (!empty($tempLogoPath) && file_exists($tempLogoPath)) {
            // Get image dimensions and calculate position for centering
            $pageWidth = $this->getPageWidth();
            $margins = $this->getMargins();
            $logoWidth = 30; // Set logo width
            $logoX = ($pageWidth - $logoWidth) / 2; // Center horizontally
            
            $this->Image($tempLogoPath, $logoX, $this->GetY(), $logoWidth, 0, '', '', '', false, 300, '', false, false, 0);
            $this->Ln(20); // Space after logo
        }
        
        // Store information below
        $this->SetFont('helvetica', 'B', 14);
        $this->Cell(0, 8, $storeName, 0, 1, 'C');
        
        $this->SetFont('helvetica', '', 10);
        $this->Cell(0, 5, $storeAddress, 0, 1, 'C');
        
        // Phone and Email on one line
        $this->Cell(0, 5, 'Phone: ' . $storePhone . ' | Email: ' . $storeEmail, 0, 1, 'C');
        
        // Website on separate line if exists
        if (!empty($storeWebsite)) {
            $this->Cell(0, 5, 'Website: ' . $storeWebsite, 0, 1, 'C');
        }
        
        $this->Ln(8);
        
        // Report period after store information
        $this->SetDrawColor(0,0,0);
        $this->Line($this->GetX(), $this->GetY(), $this->getPageWidth() - $this->GetX(), $this->GetY());
        $this->Ln(2);
        $this->SetFont('helvetica', 'B', 12);
        $this->Cell(0, 8, 'Report Period: ' . date('M d, Y', strtotime($start_date)) . ' to ' . date('M d, Y', strtotime($end_date)), 0, 1, 'C');
        $this->Line($this->GetX(), $this->GetY(), $this->getPageWidth() - $this->GetX(), $this->GetY());
        $this->Ln(30);
    }
    
    public function Footer() {
        $this->SetY(-15);
        $this->SetFont('helvetica', 'I', 8);
        $this->Cell(0, 10, 'Generated on ' . date('Y-m-d H:i:s') . ' | Page ' . $this->getAliasNumPage() . '/' . $this->getAliasNbPages(), 0, 0, 'C');
    }
}

// Create new PDF document
$pdf = new MYPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

// Set document information
$pdf->SetCreator('POS System');
$pdf->SetAuthor($storeName);
$pdf->SetTitle('Sales Report - ' . $start_date . ' to ' . $end_date);
$pdf->SetSubject('Sales Report');

// Set margins (1.5cm all around, increased top margin to accommodate larger header)
$pdf->SetMargins(15, 50, 15); // 15mm = 1.5cm
$pdf->SetHeaderMargin(5);
$pdf->SetFooterMargin(10);

// Set auto page breaks
$pdf->SetAutoPageBreak(TRUE, 25);

// Add a page
$pdf->AddPage();

// Summary statistics
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'SUMMARY STATISTICS', 0, 1, 'L');
$pdf->Ln(2);

$pdf->SetFont('helvetica', '', 10);
$pdf->Cell(50, 6, 'Total Transactions:', 0, 0, 'L');
$pdf->Cell(40, 6, number_format($stats['total_transactions']), 0, 1, 'L');

$pdf->Cell(50, 6, 'Total Revenue:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['total_revenue'], 2), 0, 1, 'L');

$pdf->Cell(50, 6, 'Average Sale:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['average_sale'], 2), 0, 1, 'L');

$pdf->Cell(50, 6, 'Cash Sales:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['cash_sales'], 2), 0, 1, 'L');

$pdf->Cell(50, 6, 'M-Pesa Sales:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['mpesa_sales'], 2), 0, 1, 'L');

$pdf->Cell(50, 6, 'Credit Sales:', 0, 0, 'L');
$pdf->Cell(40, 6, $currency . ' ' . number_format($stats['credit_sales'], 2), 0, 1, 'L');

$pdf->Ln(10);

// Sales table header
$pdf->SetFont('helvetica', 'B', 11);
$pdf->Cell(0, 8, 'DETAILED SALES TRANSACTIONS', 0, 1, 'L');
$pdf->Ln(2);

// Table header - adjusted to fit within 1.5cm margins (180mm total width)
$pdf->SetFont('helvetica', 'B', 9);
$pdf->SetFillColor(230, 230, 230);
$pdf->Cell(22, 8, 'Receipt', 1, 0, 'C', true);
$pdf->Cell(28, 8, 'Date', 1, 0, 'C', true);
$pdf->Cell(40, 8, 'Customer', 1, 0, 'C', true);
$pdf->Cell(28, 8, 'Amount', 1, 0, 'C', true);
$pdf->Cell(22, 8, 'Payment', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Status', 1, 0, 'C', true);
$pdf->Cell(20, 8, 'Cashier', 1, 1, 'C', true);

// Table data
$pdf->SetFont('helvetica', '', 8);
$pdf->SetFillColor(255, 255, 255);

foreach ($sales as $sale) {
    // Check if we need a new page
    if ($pdf->GetY() > 250) {
        $pdf->AddPage();
        // Re-add table header
        $pdf->SetFont('helvetica', 'B', 9);
        $pdf->SetFillColor(230, 230, 230);
        $pdf->Cell(22, 8, 'Receipt', 1, 0, 'C', true);
        $pdf->Cell(28, 8, 'Date', 1, 0, 'C', true);
        $pdf->Cell(40, 8, 'Customer', 1, 0, 'C', true);
        $pdf->Cell(28, 8, 'Amount', 1, 0, 'C', true);
        $pdf->Cell(22, 8, 'Payment', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Status', 1, 0, 'C', true);
        $pdf->Cell(20, 8, 'Cashier', 1, 1, 'C', true);
        $pdf->SetFont('helvetica', '', 8);
    }
    
    $pdf->Cell(22, 6, 'R' . str_pad($sale['id'], 4, '0', STR_PAD_LEFT), 1, 0, 'C');
    $pdf->Cell(28, 6, date('Y-m-d', strtotime($sale['created_at'])), 1, 0, 'C');
    $pdf->Cell(40, 6, substr($sale['customer_name'], 0, 22), 1, 0, 'L');
    $pdf->Cell(28, 6, $currency . ' ' . number_format($sale['total_amount'], 2), 1, 0, 'R');
    $pdf->Cell(22, 6, ucfirst($sale['payment_method']), 1, 0, 'C');
    $pdf->Cell(20, 6, ucfirst($sale['status']), 1, 0, 'C');
    $pdf->Cell(20, 6, substr($sale['cashier'] ?? 'N/A', 0, 10), 1, 1, 'C');
}

// Generate filename
$filename = 'Sales_Report_' . $start_date . '_to_' . $end_date . '.pdf';

// Create absolute path for reports directory
$reports_dir = dirname(__DIR__) . '/Reports/';
if (!file_exists($reports_dir)) {
    mkdir($reports_dir, 0755, true);
}

// Save PDF to reports folder (optional - for record keeping)
$filepath = $reports_dir . $filename;
try {
    $pdf->Output($filepath, 'F');
} catch (Exception $e) {
    // If saving fails, continue with download only
    error_log("Failed to save PDF to reports folder: " . $e->getMessage());
}

// Clean up temporary logo file
if ($tempLogoPath && file_exists($tempLogoPath)) {
    unlink($tempLogoPath);
}

// Clean any output buffer before sending PDF
ob_end_clean();

// Output PDF to browser for download
$pdf->Output($filename, 'D');
exit;
?> 